use std::mem;
use crate::value::Value;
use crate::build_in;
use crate::bytecode::ByteCode;
use crate::parse::Proto;
use crate::utils::{LastMap,PopX};
pub struct Execute{
    pub stack:Vec<Value>,
    pub locals:LastMap<String,Value>,
}
impl Execute{
    pub fn new()->Self{
        let mut locals:LastMap<String, Value>=LastMap::new();
        locals.insert("print".to_string(),Value::BuildInFunc(build_in::print));
        locals.insert("println".to_string(),Value::BuildInFunc(build_in::println));
        locals.insert("input".to_string(),Value::BuildInFunc(build_in::input));
        locals.insert("sin".to_string(),Value::BuildInFunc(build_in::sin));
        locals.insert("cos".to_string(),Value::BuildInFunc(build_in::cos));
        locals.insert("exit".to_string(),Value::BuildInFunc(build_in::exit));
        Self{stack:Vec::new(),locals}
    }
    pub fn exec(&mut self,proto:Proto)->Result<(),String>{
        let mut pc=0;
        while pc<proto.bytecodes.len(){
            match &proto.bytecodes[pc]{
                ByteCode::LoadValue(value)=>self.stack.push(value.clone()),
                ByteCode::Call(n)=>{
                    if let Some(Value::BuildInFunc(func))=self.locals.get(proto.names[*n].clone()){
                        func(self)
                    }else{return Err(format!("Not found function:{}",proto.names[*n]))};
                    self.stack.clear();
                }
                ByteCode::CallResult(n,arg_num)=>{
                    let args=self.stack.split_off(self.stack.len()-*arg_num as usize);
                    let old_stack=mem::replace(&mut self.stack,args);
                    let v=if let Some(Value::BuildInFunc(func))=self.locals.get(proto.names[*n].clone()){
                        func(self)
                    }else{return Err(format!("Not found function:{}",proto.names[*n]))};
                    self.stack=old_stack;
                    self.stack.push(v);
                }
                ByteCode::DefineVar(n)=>{
                    self.locals.insert(proto.names[*n].clone(),self.stack.popx());
                }
                ByteCode::Move(n)=>{
                    if let Some(value)=self.locals.get(proto.names[*n].clone()){
                        self.stack.push(value.clone());
                    }else{
                        return Err(format!("Not found var:{}",proto.names[*n]))
                    }
                }
                ByteCode::SetVar(n)=>{
                    self.locals.set(proto.names[*n].clone(),self.stack.popx());
                }
                ByteCode::IfJump(i)=>{
                    if matches!(self.stack.popx(),Value::Int(0)|Value::Bool(0)|Value::Nil){
                        pc+=*i;
                        continue;
                    }
                }
                ByteCode::Drop(i)=>{
                    self.locals.truncate(self.locals.len()-*i as usize);
                }
                ByteCode::BinaryOp(t)=>{
                    let (left,right)=(self.stack.popx(),self.stack.popx());
                    match t{
                        0=>self.stack.push((left+right)?),
                        1=>self.stack.push((left-right)?),
                        2=>self.stack.push((left*right)?),
                        3=>self.stack.push((left/right)?),
                        4=>self.stack.push(Value::Bool((left==right) as u8)),
                        5=>self.stack.push(Value::Bool((left>=right) as u8)),
                        6=>self.stack.push(Value::Bool((left>right) as u8)),
                        7=>self.stack.push(Value::Bool((left<=right) as u8)),
                        8=>self.stack.push(Value::Bool((left<right) as u8)),
                        9=>self.stack.push(left.pow(right)?),
                        _=>unreachable!(),
                    }
                }
                ByteCode::UnaryOp(t)=>{
                    let value=self.stack.popx();
                    match t{
                        0=>self.stack.push((!value)?),
                        _=>unreachable!(),
                    }
                }
            }
            pc+=1;
        }
        Ok(())
    }
}