use crate::{lex::*,bytecode::ByteCode};
use crate::ast::{AstNode,ast_type};
use std::collections::HashMap;
#[derive(Debug)]
pub struct Proto{
    pub bytecodes:Vec<ByteCode>,
    pub names:Vec<String>,
    name_map:HashMap<String,usize>,
    var_number:u16,
    function_number:u16,
}
impl Proto{
    pub fn new()->Self{
        Self{
            bytecodes:Vec::new(),
            names:Vec::new(),name_map:HashMap::new(),
            var_number:0,function_number:0
        }
    }
    pub fn parse(lex:&mut Lex)->Result<Proto,String>{
        let mut proto=Self::new();
        let ast_vec=AstNode::ast_vec(lex,None,Token::Eos)?;
        for ast in ast_vec{
            Self::statment(ast,&mut proto,false);
        }
        Ok(proto)
    }
    fn load_name(proto:&mut Proto,name:String)->usize{
        if let Some(index)=proto.name_map.get(&name){
            return *index
        }
        proto.names.push(name.clone());
        proto.name_map.insert(name,proto.names.len()-1);
        proto.names.len()-1
    }
    fn statment(ast:AstNode,proto:&mut Proto,evaluate:bool){
        match ast{
            AstNode::Value(v)=>{
                if evaluate{
                    proto.bytecodes.push(ByteCode::LoadValue(v))
                }
            }
            AstNode::Call(ast_type::CallNode{name,args})=>{
                if evaluate{Self::parse_function(name,args,true,proto)}
                else{Self::parse_function(name,args,false,proto)}
            }
            AstNode::DefineVariable(var_ast)=>{
                proto.var_number+=1;
                let ast_type::VariableNode{name,value}=*var_ast;
                Self::parse_define_or_set_var(name,value,0,proto);
            }
            AstNode::SetVariable(var_ast)=>{
                let ast_type::VariableNode{name,value}=*var_ast;
                Self::parse_define_or_set_var(name,value,1,proto);
            }
            AstNode::UseVariable(name)=>{
                if evaluate{
                    let name_index=Self::load_name(proto,name);
                    proto.bytecodes.push(ByteCode::Move(name_index));
                }
            }
            AstNode::If(if_ast)=>{
                let ast_type::IfNode{cond,block}=*if_ast;
                Self::statment(cond,proto,true);
                proto.bytecodes.push(ByteCode::IfJump(0));
                let ifjump_index=proto.bytecodes.len()-1;
                proto.bytecodes[ifjump_index]=ByteCode::IfJump(Self::block(block,proto)+1);
                proto.bytecodes.push(ByteCode::Drop(proto.function_number+proto.var_number));
            }
            AstNode::BinaryOp(binary_op)=>{
                if evaluate{
                    let ast_type::BinaryOpNode{left,op,right}=*binary_op;
                    Self::statment(right,proto,true);
                    Self::statment(left,proto,true);
                    match op{
                        Token::Plus=>proto.bytecodes.push(ByteCode::BinaryOp(0)),
                        Token::Minus=>proto.bytecodes.push(ByteCode::BinaryOp(1)),
                        Token::Mul=>proto.bytecodes.push(ByteCode::BinaryOp(2)),
                        Token::Div=>proto.bytecodes.push(ByteCode::BinaryOp(3)),
                        Token::Eq=>proto.bytecodes.push(ByteCode::BinaryOp(4)),
                        Token::Ne=>{
                            proto.bytecodes.push(ByteCode::BinaryOp(4));
                            proto.bytecodes.push(ByteCode::UnaryOp(0));
                        }
                        Token::LesEq=>proto.bytecodes.push(ByteCode::BinaryOp(7)),
                        Token::Less=>proto.bytecodes.push(ByteCode::BinaryOp(8)),
                        Token::GreEq=>proto.bytecodes.push(ByteCode::BinaryOp(5)),
                        Token::Greater=>proto.bytecodes.push(ByteCode::BinaryOp(6)),
                        Token::Pow=>proto.bytecodes.push(ByteCode::BinaryOp(9)),
                        _=>unreachable!(),
                    }
                }
            }
            AstNode::UnaryOp(unary_op)=>{
                let ast_type::UnaryOpNode{op,exp}=*unary_op;
                Self::statment(exp,proto,true);
                match op{
                    Token::Not=>proto.bytecodes.push(ByteCode::UnaryOp(0)),
                    _=>unreachable!(),
                }
            }
        }
    }
    fn parse_function(name:String,args:Vec<AstNode>,want_return:bool,proto:&mut Proto){
        let mut arg_num:u8=0;
        let name_index=Self::load_name(proto,name);
        for arg in args{
            Self::statment(arg,proto,true);
            arg_num+=1
        }
        if want_return{
            proto.bytecodes.push(ByteCode::CallResult(name_index,arg_num))
        }else{
            proto.bytecodes.push(ByteCode::Call(name_index))
        }
    }
    fn parse_define_or_set_var(name:String,value:AstNode,class:u8,proto:&mut Proto){
        let name_index=Self::load_name(proto,name);
        Self::statment(value,proto,true);
        if class==0{
            proto.bytecodes.push(ByteCode::DefineVar(name_index));
        }else if class==1{
            proto.bytecodes.push(ByteCode::SetVar(name_index));
        }
    }
    fn block(block:Vec<AstNode>,proto:&mut Proto)->usize{
        (proto.function_number,proto.var_number)=(0,0);
        let len=proto.bytecodes.len();
        for ast in block{
            Self::statment(ast,proto,false);
        }
        proto.bytecodes.len()-len
    }
}